<?php

/**
 * Quick Setup Script for MultiPOS Module Testing
 * 
 * Run this via: php artisan tinker < test_multipos_setup.php
 * Or copy and paste into tinker
 */

echo "=== MultiPOS Module Phase 1 Setup & Testing ===\n\n";

// 1. Check if table exists
echo "1. Checking database table...\n";
try {
    $tableExists = \DB::select("SHOW TABLES LIKE 'pos_machines'");
    if (count($tableExists) > 0) {
        echo "   ✓ Table 'pos_machines' exists\n";
    } else {
        echo "   ✗ Table 'pos_machines' NOT found. Please run: php artisan migrate\n";
        exit(1);
    }
} catch (\Exception $e) {
    echo "   ✗ Error checking table: " . $e->getMessage() . "\n";
    exit(1);
}

// 2. Create/Check Module
echo "\n2. Checking/Creating MultiPOS module...\n";
$module = \App\Models\Module::where('name', 'MultiPOS')->first();
if (!$module) {
    $module = \App\Models\Module::create([
        'name' => 'MultiPOS',
        'is_active' => true,
    ]);
    echo "   ✓ Module 'MultiPOS' created\n";
} else {
    echo "   ✓ Module 'MultiPOS' already exists\n";
}

// 3. Create Permission
echo "\n3. Checking/Creating permission...\n";
$permission = \Spatie\Permission\Models\Permission::where('name', 'Manage MultiPOS Machines')->first();
if (!$permission) {
    $permission = \Spatie\Permission\Models\Permission::create([
        'name' => 'Manage MultiPOS Machines',
        'guard_name' => 'web',
    ]);
    echo "   ✓ Permission 'Manage MultiPOS Machines' created\n";
} else {
    echo "   ✓ Permission 'Manage MultiPOS Machines' already exists\n";
}

// 4. Get or Create Test Data
echo "\n4. Setting up test data...\n";
$branch = \App\Models\Branch::first();
if (!$branch) {
    echo "   ✗ No branch found. Please create a branch first.\n";
    exit(1);
}
echo "   ✓ Using branch: {$branch->name}\n";

$restaurant = $branch->restaurant;
echo "   ✓ Restaurant: {$restaurant->name}\n";

$package = $restaurant->package;
if (!$package) {
    echo "   ⚠ No package assigned to restaurant. Creating default package...\n";
    $package = \App\Models\Package::firstOrCreate([
        'package_name' => 'Test Package',
        'package_type' => 'monthly',
    ]);
    $restaurant->update(['package_id' => $package->id]);
    echo "   ✓ Default package created and assigned\n";
} else {
    echo "   ✓ Package: {$package->package_name}\n";
}

// 5. Add MultiPOS to Package
echo "\n5. Adding MultiPOS to package...\n";
if (!$package->hasModule('MultiPOS')) {
    $package->modules()->syncWithoutDetaching([$module->id]);
    echo "   ✓ MultiPOS module added to package\n";
} else {
    echo "   ✓ MultiPOS module already in package\n";
}

// Set package limit if not set
if (!isset($package->multipos_limit) || $package->multipos_limit === null) {
    $package->update(['multipos_limit' => 5]);
    echo "   ✓ Package limit set to 5 machines\n";
} else {
    echo "   ✓ Package limit: " . ($package->multipos_limit == -1 ? 'Unlimited' : $package->multipos_limit) . "\n";
}

// 6. Assign Permission to Admin Role
echo "\n6. Assigning permission to roles...\n";
$adminRole = \Spatie\Permission\Models\Role::where('name', 'Admin')->first();
if (!$adminRole) {
    // Try to find any role
    $adminRole = \Spatie\Permission\Models\Role::first();
}

if ($adminRole) {
    if (!$adminRole->hasPermissionTo($permission)) {
        $adminRole->givePermissionTo($permission);
        echo "   ✓ Permission assigned to '{$adminRole->name}' role\n";
    } else {
        echo "   ✓ Permission already assigned to '{$adminRole->name}' role\n";
    }
} else {
    echo "   ⚠ No role found. Please assign permission manually.\n";
}

// 7. Test Service Methods
echo "\n7. Testing service methods...\n";
try {
    use Modules\MultiPOS\Services\PosMachineService;
    
    $canRegister = PosMachineService::canRegisterNewMachine($branch->id);
    echo "   ✓ Can register new machine: " . ($canRegister ? 'YES' : 'NO') . "\n";
    
    $count = PosMachineService::getMachineCount($branch->id);
    echo "   ✓ Current machine count: {$count}\n";
    
    $remaining = PosMachineService::getRemainingSlots($branch->id);
    echo "   ✓ Remaining slots: " . ($remaining !== null ? $remaining : 'Unlimited') . "\n";
} catch (\Exception $e) {
    echo "   ✗ Error testing service: " . $e->getMessage() . "\n";
}

// 8. Test Entity
echo "\n8. Testing entity...\n";
try {
    use Modules\MultiPOS\Entities\PosMachine;
    
    $deviceId = PosMachine::generateDeviceId();
    echo "   ✓ Generated device ID: {$deviceId}\n";
    echo "   ✓ Device ID format valid: " . (strlen($deviceId) > 0 ? 'YES' : 'NO') . "\n";
} catch (\Exception $e) {
    echo "   ✗ Error testing entity: " . $e->getMessage() . "\n";
}

echo "\n=== Setup Complete ===\n\n";
echo "Next steps:\n";
echo "1. Login as a user with 'Manage MultiPOS Machines' permission\n";
echo "2. Navigate to: http://localhost/restorank/public/multi-pos/settings\n";
echo "3. Test creating machines via tinker (see testing guide)\n";
echo "\n";

