<?php

namespace Modules\MultiPOS\Services;

use Modules\MultiPOS\Entities\PosMachine;
use App\Models\Order;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class PosMachineUsageService
{
    /**
     * Update last seen timestamp for a machine
     */
    public static function updateLastSeen(string $deviceId, int $branchId): void
    {
        $machine = PosMachine::where('device_id', $deviceId)
            ->where('branch_id', $branchId)
            ->first();

        if ($machine) {
            $machine->updateLastSeen();
        }
    }

    /**
     * Get usage statistics for a machine
     */
    public static function getMachineStats(int $machineId, ?Carbon $startDate = null, ?Carbon $endDate = null): array
    {
        $startDate = $startDate ?? Carbon::now()->startOfMonth();
        $endDate = $endDate ?? Carbon::now()->endOfMonth();

        $orders = Order::where('pos_machine_id', $machineId)
            ->whereBetween('date_time', [$startDate, $endDate])
            ->get();

        $totalOrders = $orders->count();
        $netSales = $orders->sum('grand_total');
        $avgOrderValue = $totalOrders > 0 ? $netSales / $totalOrders : 0;

        // Cash vs Card/UPI breakdown
        $cashSales = 0;
        $cardSales = 0;

        foreach ($orders as $order) {
            $payments = $order->payments ?? collect();
            foreach ($payments as $payment) {
                $amount = (float) ($payment->amount ?? 0);
                if (($payment->payment_method ?? null) === 'cash') {
                    $cashSales += $amount;
                } else {
                    $cardSales += $amount;
                }
            }
        }

        return [
            'total_orders' => $totalOrders,
            'net_sales' => $netSales,
            'avg_order_value' => $avgOrderValue,
            'cash_sales' => $cashSales,
            'card_sales' => $cardSales,
        ];
    }

    /**
     * Get usage statistics for all machines in a branch
     */
    public static function getBranchStats(int $branchId, ?Carbon $startDate = null, ?Carbon $endDate = null): array
    {
        $machines = PosMachine::where('branch_id', $branchId)
            ->where('status', 'active')
            ->get();

        $stats = [];
        foreach ($machines as $machine) {
            $stats[$machine->id] = self::getMachineStats($machine->id, $startDate, $endDate);
            $stats[$machine->id]['machine'] = $machine;
        }

        return $stats;
    }
}

