<?php

namespace Modules\Inventory\Tests\Unit;

use Tests\TestCase;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Modules\Inventory\Entities\InventoryItem;
use Modules\Inventory\Entities\InventoryStock;
use Modules\Inventory\Entities\InventoryItemCategory;
use Modules\Inventory\Entities\Unit;
use App\Models\Branch;
use App\Models\Restaurant;
use App\Models\User;

class InventoryItemTest extends TestCase
{
    use RefreshDatabase;

    protected $restaurant;
    protected $branch;
    protected $category;
    protected $unit;

    protected function setUp(): void
    {
        parent::setUp();

        // Create test data
        $this->restaurant = Restaurant::factory()->create();
        $this->branch = Branch::factory()->create(['restaurant_id' => $this->restaurant->id]);
        $this->category = InventoryItemCategory::factory()->create(['branch_id' => $this->branch->id]);
        $this->unit = Unit::factory()->create(['branch_id' => $this->branch->id]);
    }

    /** @test */
    public function it_calculates_current_stock_correctly()
    {
        $item = InventoryItem::factory()->create([
            'branch_id' => $this->branch->id,
            'inventory_item_category_id' => $this->category->id,
            'unit_id' => $this->unit->id,
        ]);

        // Add stock entries
        InventoryStock::factory()->create([
            'inventory_item_id' => $item->id,
            'branch_id' => $this->branch->id,
            'quantity' => 10.5,
        ]);

        InventoryStock::factory()->create([
            'inventory_item_id' => $item->id,
            'branch_id' => $this->branch->id,
            'quantity' => 5.25,
        ]);

        // Refresh to load relationships
        $item->refresh();

        $this->assertEquals(15.75, $item->current_stock);
    }

    /** @test */
    public function it_returns_zero_stock_when_no_stocks_exist()
    {
        $item = InventoryItem::factory()->create([
            'branch_id' => $this->branch->id,
            'inventory_item_category_id' => $this->category->id,
            'unit_id' => $this->unit->id,
        ]);

        $item->refresh();

        $this->assertEquals(0, $item->current_stock);
    }

    /** @test */
    public function it_returns_out_of_stock_status_when_stock_is_zero()
    {
        $item = InventoryItem::factory()->create([
            'branch_id' => $this->branch->id,
            'inventory_item_category_id' => $this->category->id,
            'unit_id' => $this->unit->id,
            'threshold_quantity' => 10,
        ]);

        $status = $item->getStockStatus();

        $this->assertEquals('Out of Stock', $status['status']);
        $this->assertStringContainsString('red', $status['class']);
    }

    /** @test */
    public function it_returns_low_stock_status_when_below_threshold()
    {
        $item = InventoryItem::factory()->create([
            'branch_id' => $this->branch->id,
            'inventory_item_category_id' => $this->category->id,
            'unit_id' => $this->unit->id,
            'threshold_quantity' => 10,
        ]);

        InventoryStock::factory()->create([
            'inventory_item_id' => $item->id,
            'branch_id' => $this->branch->id,
            'quantity' => 5, // Below threshold of 10
        ]);

        $item->refresh();
        $status = $item->getStockStatus();

        $this->assertEquals('Low Stock', $status['status']);
        $this->assertStringContainsString('yellow', $status['class']);
    }

    /** @test */
    public function it_returns_in_stock_status_when_above_threshold()
    {
        $item = InventoryItem::factory()->create([
            'branch_id' => $this->branch->id,
            'inventory_item_category_id' => $this->category->id,
            'unit_id' => $this->unit->id,
            'threshold_quantity' => 10,
        ]);

        InventoryStock::factory()->create([
            'inventory_item_id' => $item->id,
            'branch_id' => $this->branch->id,
            'quantity' => 25, // Above threshold of 10
        ]);

        $item->refresh();
        $status = $item->getStockStatus();

        $this->assertEquals('In Stock', $status['status']);
        $this->assertStringContainsString('green', $status['class']);
    }

    /** @test */
    public function it_returns_in_stock_when_at_threshold()
    {
        $item = InventoryItem::factory()->create([
            'branch_id' => $this->branch->id,
            'inventory_item_category_id' => $this->category->id,
            'unit_id' => $this->unit->id,
            'threshold_quantity' => 10,
        ]);

        InventoryStock::factory()->create([
            'inventory_item_id' => $item->id,
            'branch_id' => $this->branch->id,
            'quantity' => 10, // Exactly at threshold
        ]);

        $item->refresh();
        $status = $item->getStockStatus();

        // At threshold should still be considered "In Stock" (not "Low Stock")
        $this->assertEquals('In Stock', $status['status']);
    }

    /** @test */
    public function it_handles_negative_stock_correctly()
    {
        $item = InventoryItem::factory()->create([
            'branch_id' => $this->branch->id,
            'inventory_item_category_id' => $this->category->id,
            'unit_id' => $this->unit->id,
        ]);

        InventoryStock::factory()->create([
            'inventory_item_id' => $item->id,
            'branch_id' => $this->branch->id,
            'quantity' => -5, // Negative stock
        ]);

        $item->refresh();
        $status = $item->getStockStatus();

        // Negative stock should be considered "Out of Stock"
        $this->assertEquals('Out of Stock', $status['status']);
    }

    /** @test */
    public function it_has_category_relationship()
    {
        $item = InventoryItem::factory()->create([
            'branch_id' => $this->branch->id,
            'inventory_item_category_id' => $this->category->id,
            'unit_id' => $this->unit->id,
        ]);

        $this->assertInstanceOf(InventoryItemCategory::class, $item->category);
        $this->assertEquals($this->category->id, $item->category->id);
    }

    /** @test */
    public function it_has_unit_relationship()
    {
        $item = InventoryItem::factory()->create([
            'branch_id' => $this->branch->id,
            'inventory_item_category_id' => $this->category->id,
            'unit_id' => $this->unit->id,
        ]);

        $this->assertInstanceOf(Unit::class, $item->unit);
        $this->assertEquals($this->unit->id, $item->unit->id);
    }

    /** @test */
    public function it_has_stocks_relationship()
    {
        $item = InventoryItem::factory()->create([
            'branch_id' => $this->branch->id,
            'inventory_item_category_id' => $this->category->id,
            'unit_id' => $this->unit->id,
        ]);

        InventoryStock::factory()->count(3)->create([
            'inventory_item_id' => $item->id,
            'branch_id' => $this->branch->id,
        ]);

        $item->refresh();

        $this->assertCount(3, $item->stocks);
    }

    /** @test */
    public function it_only_sums_stocks_for_current_item()
    {
        $item1 = InventoryItem::factory()->create([
            'branch_id' => $this->branch->id,
            'inventory_item_category_id' => $this->category->id,
            'unit_id' => $this->unit->id,
        ]);

        $item2 = InventoryItem::factory()->create([
            'branch_id' => $this->branch->id,
            'inventory_item_category_id' => $this->category->id,
            'unit_id' => $this->unit->id,
        ]);

        // Add stocks to both items
        InventoryStock::factory()->create([
            'inventory_item_id' => $item1->id,
            'branch_id' => $this->branch->id,
            'quantity' => 10,
        ]);

        InventoryStock::factory()->create([
            'inventory_item_id' => $item2->id,
            'branch_id' => $this->branch->id,
            'quantity' => 20,
        ]);

        $item1->refresh();
        $item2->refresh();

        // Each item should only sum its own stocks
        $this->assertEquals(10, $item1->current_stock);
        $this->assertEquals(20, $item2->current_stock);
    }
}

