<?php

namespace Modules\CashRegister\Tests\Unit;

use Tests\TestCase;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Modules\CashRegister\Entities\CashRegister;
use Modules\CashRegister\Entities\CashRegisterSession;
use Modules\CashRegister\Entities\CashRegisterTransaction;
use App\Models\Branch;
use App\Models\Restaurant;
use App\Models\User;
use Carbon\Carbon;

class CashRegisterSessionTest extends TestCase
{
    use RefreshDatabase;

    protected $restaurant;
    protected $branch;
    protected $cashRegister;
    protected $user;

    protected function setUp(): void
    {
        parent::setUp();

        // Create test data
        $this->restaurant = Restaurant::factory()->create();
        $this->branch = Branch::factory()->create(['restaurant_id' => $this->restaurant->id]);
        $this->user = User::factory()->create(['restaurant_id' => $this->restaurant->id, 'branch_id' => $this->branch->id]);
        $this->cashRegister = CashRegister::factory()->create([
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
        ]);
    }

    /** @test */
    public function it_calculates_expected_cash_correctly()
    {
        $session = CashRegisterSession::factory()->create([
            'cash_register_id' => $this->cashRegister->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'opened_by' => $this->user->id,
            'opening_float' => 100.00,
            'status' => 'open',
        ]);

        // Add transactions
        CashRegisterTransaction::factory()->create([
            'cash_register_session_id' => $session->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'type' => 'cash_sale',
            'amount' => 50.00,
            'created_by' => $this->user->id,
        ]);

        CashRegisterTransaction::factory()->create([
            'cash_register_session_id' => $session->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'type' => 'cash_in',
            'amount' => 25.00,
            'created_by' => $this->user->id,
        ]);

        CashRegisterTransaction::factory()->create([
            'cash_register_session_id' => $session->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'type' => 'change_given',
            'amount' => 5.00,
            'created_by' => $this->user->id,
        ]);

        CashRegisterTransaction::factory()->create([
            'cash_register_session_id' => $session->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'type' => 'cash_out',
            'amount' => 10.00,
            'created_by' => $this->user->id,
        ]);

        $session->refresh();

        // Expected = Opening Float + Cash Sales + Cash In - Change Given - Cash Out
        // Expected = 100 + 50 + 25 - 5 - 10 = 160
        $expectedCash = 100.00 + 50.00 + 25.00 - 5.00 - 10.00;

        $cashSales = $session->transactions()->where('type', 'cash_sale')->sum('amount');
        $cashIn = $session->transactions()->where('type', 'cash_in')->sum('amount');
        $changeGiven = $session->transactions()->where('type', 'change_given')->sum('amount');
        $cashOut = $session->transactions()->where('type', 'cash_out')->sum('amount');

        $calculated = $session->opening_float + $cashSales + $cashIn - $changeGiven - $cashOut;

        $this->assertEquals($expectedCash, $calculated);
    }

    /** @test */
    public function it_calculates_discrepancy_correctly()
    {
        $session = CashRegisterSession::factory()->create([
            'cash_register_id' => $this->cashRegister->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'opened_by' => $this->user->id,
            'opening_float' => 100.00,
            'expected_cash' => 160.00,
            'counted_cash' => 155.00, // $5 short
            'discrepancy' => 0,
            'status' => 'open',
        ]);

        // Discrepancy = Counted - Expected
        $discrepancy = $session->counted_cash - $session->expected_cash;

        $this->assertEquals(-5.00, $discrepancy);
    }

    /** @test */
    public function it_handles_zero_opening_float()
    {
        $session = CashRegisterSession::factory()->create([
            'cash_register_id' => $this->cashRegister->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'opened_by' => $this->user->id,
            'opening_float' => 0.00,
            'status' => 'open',
        ]);

        CashRegisterTransaction::factory()->create([
            'cash_register_session_id' => $session->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'type' => 'cash_sale',
            'amount' => 50.00,
            'created_by' => $this->user->id,
        ]);

        $session->refresh();

        $cashSales = $session->transactions()->where('type', 'cash_sale')->sum('amount');
        $expectedCash = $session->opening_float + $cashSales;

        $this->assertEquals(50.00, $expectedCash);
    }

    /** @test */
    public function it_has_transactions_relationship()
    {
        $session = CashRegisterSession::factory()->create([
            'cash_register_id' => $this->cashRegister->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'opened_by' => $this->user->id,
            'status' => 'open',
        ]);

        CashRegisterTransaction::factory()->count(5)->create([
            'cash_register_session_id' => $session->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'created_by' => $this->user->id,
        ]);

        $session->refresh();

        $this->assertCount(5, $session->transactions);
    }

    /** @test */
    public function it_filters_transactions_by_type_correctly()
    {
        $session = CashRegisterSession::factory()->create([
            'cash_register_id' => $this->cashRegister->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'opened_by' => $this->user->id,
            'status' => 'open',
        ]);

        // Create different transaction types
        CashRegisterTransaction::factory()->count(3)->create([
            'cash_register_session_id' => $session->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'type' => 'cash_sale',
            'amount' => 10.00,
            'created_by' => $this->user->id,
        ]);

        CashRegisterTransaction::factory()->count(2)->create([
            'cash_register_session_id' => $session->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'type' => 'cash_out',
            'amount' => 5.00,
            'created_by' => $this->user->id,
        ]);

        $session->refresh();

        $cashSales = $session->transactions()->where('type', 'cash_sale')->get();
        $cashOut = $session->transactions()->where('type', 'cash_out')->get();

        $this->assertCount(3, $cashSales);
        $this->assertCount(2, $cashOut);

        $totalSales = $cashSales->sum('amount');
        $totalOut = $cashOut->sum('amount');

        $this->assertEquals(30.00, $totalSales);
        $this->assertEquals(10.00, $totalOut);
    }

    /** @test */
    public function it_handles_safe_drops_and_refunds()
    {
        $session = CashRegisterSession::factory()->create([
            'cash_register_id' => $this->cashRegister->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'opened_by' => $this->user->id,
            'opening_float' => 100.00,
            'status' => 'open',
        ]);

        CashRegisterTransaction::factory()->create([
            'cash_register_session_id' => $session->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'type' => 'cash_sale',
            'amount' => 100.00,
            'created_by' => $this->user->id,
        ]);

        CashRegisterTransaction::factory()->create([
            'cash_register_session_id' => $session->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'type' => 'safe_drop',
            'amount' => 50.00, // Removed from drawer
            'created_by' => $this->user->id,
        ]);

        CashRegisterTransaction::factory()->create([
            'cash_register_session_id' => $session->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'type' => 'refund',
            'amount' => 10.00, // Refunded to customer
            'created_by' => $this->user->id,
        ]);

        $session->refresh();

        // Expected = Opening + Sales - Safe Drop - Refund
        // Expected = 100 + 100 - 50 - 10 = 140
        $cashSales = $session->transactions()->where('type', 'cash_sale')->sum('amount');
        $safeDrops = $session->transactions()->where('type', 'safe_drop')->sum('amount');
        $refunds = $session->transactions()->where('type', 'refund')->sum('amount');

        $expectedCash = $session->opening_float + $cashSales - $safeDrops - $refunds;

        $this->assertEquals(140.00, $expectedCash);
    }

    /** @test */
    public function it_has_register_relationship()
    {
        $session = CashRegisterSession::factory()->create([
            'cash_register_id' => $this->cashRegister->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'opened_by' => $this->user->id,
            'status' => 'open',
        ]);

        $this->assertInstanceOf(CashRegister::class, $session->register);
        $this->assertEquals($this->cashRegister->id, $session->register->id);
    }

    /** @test */
    public function it_has_branch_relationship()
    {
        $session = CashRegisterSession::factory()->create([
            'cash_register_id' => $this->cashRegister->id,
            'restaurant_id' => $this->restaurant->id,
            'branch_id' => $this->branch->id,
            'opened_by' => $this->user->id,
            'status' => 'open',
        ]);

        $this->assertInstanceOf(Branch::class, $session->branch);
        $this->assertEquals($this->branch->id, $session->branch->id);
    }
}

